﻿using System;
using System.Collections.Generic;
using System.Linq;
using NVCC.WebUI.Models;
using NVCC.Models;
using NVCC.Repos.AodRepository;

namespace NVCC.WebUI.Infrastructure
{
    public class AodService : IAodService
    {
        private readonly IAodRepository _aodRepository;

        private static readonly Nature[] _allNatures = new AodContext().Natures.ToArray();

        //Using "Single" rather than "First or Default" here because, there should be exactly One match for each of these in the Nature Table.
        //If not, something's wrong, and we'll get an error immediately.
        private static readonly Nature _PatientNameSsnDobNature = _allNatures.Single(n => n.NatureName == "Patient name, SSN, and DOB");
        private static readonly Nature _ReferralTypeNature = _allNatures.Single(n => n.NatureName == "Referral Type");
        private static readonly Nature _AuthorizationsNature = _allNatures.Single(n => n.NatureName == "Authorizations");
        private static readonly Nature _ConsultsNature = _allNatures.Single(n => n.NatureName == "Consults");
        private static readonly Nature _OrdersNature = _allNatures.Single(n => n.NatureName == "Orders");
        private static readonly Nature _MedicationListNature = _allNatures.Single(n => n.NatureName == "Medication list");
        private static readonly Nature _AllergiesNature = _allNatures.Single(n => n.NatureName == "Allergies");
        private static readonly Nature _RadiologyNature = _allNatures.Single(n => n.NatureName == "Radiology");
        private static readonly Nature _ProgressNotesNature = _allNatures.Single(n => n.NatureName == "Progress Notes");
        private static readonly Nature _ProblemListNature = _allNatures.Single(n => n.NatureName == "Problem list");
        private static readonly Nature _AppointmentListNature = _allNatures.Single(n => n.NatureName == "Appointment list");
        private static readonly Nature _LabsNature = _allNatures.Single(n => n.NatureName == "Labs");

        //IMPORTANT: Should change this to "Single" like the others above once "VistA Imaging" is added to the Nature table
        private static readonly Nature _VistAImagesNature = _allNatures.FirstOrDefault(n => n.NatureName == "VistA Imaging");


        public AodService(IAodRepository aodRepository)
        {
            _aodRepository = aodRepository;
        }

        public void RecordAodInfo(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.DisclosureInfo == null)
            {
                throw new ApplicationException("There is no Accounting of Disclosure Info in the PatientProfileViewModel object");
            }

            patientProfile.DisclosureInfo.DateTimeOfDisclosure = DateTime.Now;
            patientProfile.DisclosureInfo.PatientIcn = patientProfile.PatientIcn;
            patientProfile.DisclosureInfo.ReferralType = patientProfile.ReferralType;

            //Remove additional objects with null or empty values, if they exist in the list
            HandleAdditionalObjects(patientProfile);

            AddNatures(patientProfile);

            AddAuthorizations(patientProfile);

            AddAppointments(patientProfile);

            //According to current requirements, we are not
            //saving Problems/Diagnoses, even sensitive ones. 
            //I'm leaving the table in the database as well as the properties in the model and the method below in.
            //Just commenting out the calling of the method here:
            //AddProblemDiagnoses(patientProfile);

            AddNotes(patientProfile);

            AddRadiology(patientProfile);

            AddConsult(patientProfile);

            if (patientProfile.IncludeVistAImaging)
            {
                AddVistAImagingStudies(patientProfile);
            }

            _aodRepository.SaveAodInfo(patientProfile.DisclosureInfo);

        }

        private void HandleAdditionalObjects(PatientProfileViewModel patientProfile)
        {
            if ((patientProfile.DisclosureInfo.AdditionalObjects != null) && (patientProfile.DisclosureInfo.AdditionalObjects.Any(item => item.ObjectDateString == null) || patientProfile.DisclosureInfo.AdditionalObjects.Any(item => item.ObjectType == null)))
            {
                IList<AdditionalObject> objects = new List<AdditionalObject>();
                foreach (AdditionalObject obj in patientProfile.DisclosureInfo.AdditionalObjects)
                {
                    if (obj.ObjectDateString != null && obj.ObjectType != null)
                    {
                        obj.DateOfObject = Convert.ToDateTime(obj.ObjectDateString);
                        objects.Add(obj);
                    }
                }
                patientProfile.DisclosureInfo.AdditionalObjects = objects;
            }
            else if (patientProfile.DisclosureInfo.AdditionalObjects != null && patientProfile.DisclosureInfo.AdditionalObjects.Any())
            {
                foreach (var obj in patientProfile.DisclosureInfo.AdditionalObjects)
                {
                    obj.DateOfObject = Convert.ToDateTime(obj.ObjectDateString);
                }
            }
        }

        private void AddNatures(PatientProfileViewModel patientProfile)
        {
            var natures = new List<Nature>();
            natures.Add(_PatientNameSsnDobNature);
            
            if (!string.IsNullOrWhiteSpace(patientProfile.ReferralType))
            {
                natures.Add(_ReferralTypeNature);
            }

            if (patientProfile.AuthorizationSelected.HasValue || !string.IsNullOrWhiteSpace(patientProfile.AdditionalAuthorizations))
            {
                natures.Add(_AuthorizationsNature);
            }

            if (patientProfile.ConsultSelected.HasValue || !string.IsNullOrWhiteSpace(patientProfile.AdditionalConsults))
            {
                natures.Add(_ConsultsNature);
            }

            if (!patientProfile.ReportOptions.OmitProblemList && patientProfile.ProblemDiagnosesSelected != null && patientProfile.ProblemDiagnosesSelected.Any(p => p))
            {
                natures.Add(_ProblemListNature);
            }

            if (!patientProfile.ReportOptions.OmitApptList && patientProfile.AppointmentsSelected != null && patientProfile.AppointmentsSelected.Any(a => a))
            {
                natures.Add(_AppointmentListNature);
            }

            if ((patientProfile.PatientProfile.Medications != null && patientProfile.PatientProfile.Medications.Any()) 
                || (patientProfile.PatientProfile.NonVaMedications != null && patientProfile.PatientProfile.NonVaMedications.Any()))
            {
                natures.Add(_MedicationListNature);
            }

            if (patientProfile.PatientProfile.Allergies != null && patientProfile.PatientProfile.Allergies.Any())
            {
                natures.Add(_AllergiesNature);
            }

            if (!string.IsNullOrWhiteSpace(patientProfile.AdditionalOrders))
            {
                natures.Add(_OrdersNature);
            }

            if (!string.IsNullOrWhiteSpace(patientProfile.ProgressNotes) || (patientProfile.NotesSelected != null && patientProfile.NotesSelected.Any(n => n)))
            {
                natures.Add(_ProgressNotesNature);
            }

            if (!patientProfile.ReportOptions.OmitLabList 
                && (!string.IsNullOrWhiteSpace(patientProfile.AdditionalLabs) || (patientProfile.PatientProfile.Labs != null && patientProfile.PatientProfile.Labs.Any())))
            {
                natures.Add(_LabsNature);
            }

            if (patientProfile.VistAImagingStudiesSelected != null && patientProfile.VistAImagingStudiesSelected.Any(v => v))
            {
                if (_VistAImagesNature != null)
                {
                    natures.Add(_VistAImagesNature);
                }
            }

            if (!string.IsNullOrWhiteSpace(patientProfile.AdditionalRadiology) || (patientProfile.RadiologyReportsSelected != null && patientProfile.RadiologyReportsSelected.Any()))
            {
                natures.Add(_RadiologyNature);
            }

            patientProfile.DisclosureInfo.Natures = natures;

        }

        private void AddAuthorizations(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.PatientProfile.Authorizations != null && patientProfile.AuthorizationSelected != null)
            {
                //I just realized that maybe there will only ever be one authorization selected so, maybe, it's a one-to-one between Disclosure and DisclosedAuthorization.
                //Because I haven't seen what it looks like when a patient has authorizations, I was unsure.
                var disclosedAuthorizations = new List<DisclosedAuthorization>();
                var authorization = new DisclosedAuthorization(patientProfile.PatientProfile.Authorizations[(int)patientProfile.AuthorizationSelected]);
                disclosedAuthorizations.Add(authorization);
                patientProfile.DisclosureInfo.DisclosedAuthorizations = disclosedAuthorizations;
            }
        }

        private void AddAppointments(PatientProfileViewModel patientProfile)
        {
            if (!patientProfile.ReportOptions.OmitApptList && patientProfile.PatientProfile.Appointments != null && patientProfile.PatientProfile.Appointments.Count > 0 && patientProfile.AppointmentsSelected != null && patientProfile.AppointmentsSelected.Count > 0)
            {
                int appointmentCount = patientProfile.AppointmentsSelected.Count(selected => selected);
                if (appointmentCount > 0)
                {
                    var disclosedAppointments = new List<DisclosedAppointment>();
                    for (int i = 0; i < patientProfile.PatientProfile.Appointments.Count(); i++)
                    {
                        if (patientProfile.AppointmentsSelected[i])
                        {
                            disclosedAppointments.Add(new DisclosedAppointment(patientProfile.PatientProfile.Appointments[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedAppointments = disclosedAppointments;
                }
            }
        }

        private void AddVistAImagingStudies(PatientProfileViewModel patientProfile)
        {
            int studiesCount = patientProfile.PatientProfile.VistAImagingStudies.Count();

            if (studiesCount > 0)
            {
                var disclosedVistAImage = new List<DisclosedVistAImagingStudy>();

                for (int i = 0; i < patientProfile.PatientProfile.VistAImagingStudies.Count(); i++)
                {
                    if (patientProfile.PatientProfile.VistAImagingStudies[i].Files != null && patientProfile.PatientProfile.VistAImagingStudies[i].Files.Any(f => f != null && f.FileBytes != null))
                    {
                        disclosedVistAImage.Add(new DisclosedVistAImagingStudy(patientProfile.PatientProfile.VistAImagingStudies[i]));
                    }
                }
                patientProfile.DisclosureInfo.DisclosedVistAImagingStudies = disclosedVistAImage;
            }
        }
        private void AddProblemDiagnoses(PatientProfileViewModel patientProfile)
        {
            if (!patientProfile.ReportOptions.OmitProblemList && patientProfile.PatientProfile.ProblemDiagnoses != null && patientProfile.PatientProfile.ProblemDiagnoses.Count > 0 && patientProfile.ProblemDiagnosesSelected != null && patientProfile.ProblemDiagnosesSelected.Count > 0)
            {
                int problemCount = patientProfile.ProblemDiagnosesSelected.Count(selected => selected);
                if (problemCount > 0)
                {
                    var disclosedProblems = new List<DisclosedProblemDiagnosis>();

                    for (int i = 0; i < patientProfile.PatientProfile.ProblemDiagnoses.Count(); i++)
                    {
                        if (patientProfile.ProblemDiagnosesSelected[i])
                        {
                            disclosedProblems.Add(new DisclosedProblemDiagnosis(patientProfile.PatientProfile.ProblemDiagnoses[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedProblemDiagnoses = disclosedProblems;
                }
            }
        }

        private void AddNotes(PatientProfileViewModel patientProfile)
        {
            var disclosedNotes = new List<DisclosedProgressNote>();

            if (patientProfile.PatientProfile.Notes != null && patientProfile.PatientProfile.Notes.Count > 0 && patientProfile.NotesSelected != null && patientProfile.NotesSelected.Count > 0)
            {
                int NotesCount = patientProfile.NotesSelected.Count(selected => selected);


                if (NotesCount > 0)
                {

                    for (int i = 0; i < patientProfile.PatientProfile.Notes.Count(); i++)
                    {
                        if (patientProfile.NotesSelected[i])
                        {
                            disclosedNotes.Add(new DisclosedProgressNote(patientProfile.PatientProfile.Notes[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedProgressNotes = disclosedNotes;
                }
            }

            //Save the opioid note if the user deceide to disclose.
            if (!patientProfile.ReportOptions.OmitOpioidList && patientProfile.PatientProfile.OpioidExist)
            {

                disclosedNotes.Add(patientProfile.PatientProfile.OpioidNote);
                patientProfile.DisclosureInfo.DisclosedProgressNotes = disclosedNotes;


            }
        }

        private void AddRadiology(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.PatientProfile.RadiologyReports != null && patientProfile.PatientProfile.RadiologyReports.Count > 0 && patientProfile.RadiologyReportsSelected != null && patientProfile.RadiologyReportsSelected.Count > 0)
            {
                int radiologyReportsCount = patientProfile.RadiologyReportsSelected.Count(selected => selected);

                if (radiologyReportsCount > 0)
                {
                    var disclosedRadiology = new List<DisclosedRadiologyReport>();
                    for (int i = 0; i < patientProfile.PatientProfile.RadiologyReports.Count(); i++)
                    {
                        if (patientProfile.RadiologyReportsSelected[i])
                        {
                            disclosedRadiology.Add(new DisclosedRadiologyReport(patientProfile.PatientProfile.RadiologyReports[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedRadiologyReports = disclosedRadiology;
                }
            }
        }



        private void AddConsult(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.PatientProfile.Consults != null && patientProfile.ConsultSelected != null)
            {
                patientProfile.DisclosureInfo.DisclosedConsults = new List<DisclosedConsult>();
                var consult = new DisclosedConsult(patientProfile.PatientProfile.Consults[(int)patientProfile.ConsultSelected]);
                patientProfile.DisclosureInfo.DisclosedConsults.Add(consult);
            }
        }


    }
}